// roles.js

// Global variables
let roles = [];
let permissions = [];
let editingRoleId = null;

// API Configuration
const API_BASE_URL = window.location.origin; // Adjust this to your API base URL
const API_ENDPOINTS = {
    roles: `${API_BASE_URL}/roles`,
    permissions: `${API_BASE_URL}/permissions`
};

// DOM elements
const rolesContainer = document.getElementById('roles-container');
const roleModal = document.getElementById('role-modal');
const roleForm = document.getElementById('role-form');
const modalTitle = document.getElementById('modal-title');
const submitBtn = document.getElementById('submit-btn');
const loadingIndicator = document.getElementById('loading-indicator');
const successNotification = document.getElementById('success-notification');
const errorNotification = document.getElementById('error-notification');

// API Helper functions
async function apiRequest(url, options = {}) {
    try {
        const config = {
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content'),
                ...options.headers
            },
            ...options
        };

        const response = await fetch(url, config);

        if (!response.ok) {
            const errorData = await response.json().catch(() => ({}));
            throw new Error(errorData.message || `HTTP error! status: ${response.status}`);
        }

        const contentType = response.headers.get('content-type');
        if (contentType && contentType.includes('application/json')) {
            return await response.json();
        }

        return await response.text();
    } catch (error) {
        console.error('API request failed:', error);
        throw error;
    }
}

// API functions for data fetching
async function fetchPermissions() {
    try {
        showLoading();
        const data = await apiRequest(API_ENDPOINTS.permissions);
        permissions = Array.isArray(data) ? data : (data.permissions || []);
        return permissions;
    } catch (error) {
        console.error('Failed to load permissions:', error);
        showNotification('error', 'Failed to load permissions: ' + error.message);
        // Fallback to empty array
        permissions = [];
        return permissions;
    } finally {
        hideLoading();
    }
}

async function fetchRoles() {
    try {
        showLoading();
        const data = await apiRequest(API_ENDPOINTS.roles);
        roles = Array.isArray(data) ? data : (data.roles || []);
        return roles;
    } catch (error) {
        console.error('Failed to load roles:', error);
        showNotification('error', 'Failed to load roles: ' + error.message);
        // Fallback to empty array
        roles = [];
        return roles;
    } finally {
        hideLoading();
    }
}

async function createRole(roleData) {
    return await apiRequest(API_ENDPOINTS.roles, {
        method: 'POST',
        body: JSON.stringify(roleData)
    });
}

async function updateRole(roleId, roleData) {
    return await apiRequest(`${API_ENDPOINTS.roles}/${roleId}`, {
        method: 'PUT',
        body: JSON.stringify(roleData)
    });
}

async function deleteRoleAPI(roleId) {
    return await apiRequest(`${API_ENDPOINTS.roles}/${roleId}`, {
        method: 'DELETE'
    });
}

// Utility functions
function showLoading() {
    if (loadingIndicator) {
        loadingIndicator.style.display = 'flex';
    }
}

function hideLoading() {
    if (loadingIndicator) {
        loadingIndicator.style.display = 'none';
    }
}

function showNotification(type, message) {
    const notification = type === 'success' ? successNotification : errorNotification;
    if (!notification) return;

    const messageElement = document.getElementById(type + '-message');
    if (messageElement) {
        messageElement.textContent = message;
    }

    notification.style.display = 'flex';
    setTimeout(() => {
        notification.style.display = 'none';
    }, 4000);
}

function formatDate(dateString) {
    if (!dateString) return 'N/A';
    try {
        return new Date(dateString).toLocaleDateString();
    } catch (error) {
        return 'Invalid Date';
    }
}

function canDeleteRole(role) {
    if (!role) return false;
    if (role.system_role) return false;

    // Use the users_count column from the role object
    const userCount = role.users_count || 0;
    return userCount === 0;
}

// Role card rendering
function createRoleCard(role, editable = false, deletable = false) {
    if (!role) return '';

    // Use users_count from the role object
    const userCount = role.users_count || 0;

    // Safely get role permissions
    const rolePermissionIds = (role.permissions || []).map(rp => rp.id);

    const rolePermissions = (Array.isArray(permissions) ? permissions : [])
        .filter(p => p && rolePermissionIds.includes(p.id));

    const canDelete = canDeleteRole(role);

    return `
        <div class="role-card bg-white rounded-lg p-6">
            <div class="flex justify-between items-start mb-4">
                <div>
                    <h3 class="text-lg font-semibold text-gray-900">${role.name || 'Unnamed Role'}</h3>
                    <p class="text-sm text-gray-600 mt-1">${role.description || 'No description provided'}</p>
                </div>
                <div class="flex items-center space-x-2">
                    <span class="user-count-badge">${userCount}</span>
                </div>
            </div>

            <div class="mb-4">
                <div class="text-sm font-medium text-gray-700 mb-2">Permissions (${rolePermissions.length})</div>
                <div class="flex flex-wrap max-h-20 overflow-hidden">
                    ${rolePermissions.slice(0, 6).map(permission =>
        `<span class="permission-tag">${permission.display_name || permission.name || 'Unknown Permission'}</span>`
    ).join('')}
                    ${rolePermissions.length > 6 ? `<span class="permission-tag">+${rolePermissions.length - 6} more</span>` : ''}
                </div>
            </div>

            <div class="text-xs text-gray-500 mb-4">
                Created: ${formatDate(role.created_at)}
                ${role.system_role ? ' • System Role' : ''}
            </div>

            <div class="flex justify-between items-center">
                <div class="text-sm text-gray-600">
                    <span class="font-medium">${userCount}</span> user${userCount !== 1 ? 's' : ''} assigned
                </div>
                <div class="flex space-x-2">
                    ${editable ? `<button onclick="editRole(${role.id})" class="px-3 py-1 bg-blue-600 text-white rounded hover:bg-blue-700 transition-colors text-sm">
                        <svg class="w-3 h-3 mr-1 inline" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                        </svg>
                        Edit
                    </button>` : ''}
                    ${deletable ? `<button onclick="deleteRole(${role.id})"
                        class="px-3 py-1 text-white rounded transition-colors text-sm ${canDelete ? 'bg-red-600 hover:bg-red-700' : 'bg-gray-400 delete-disabled'}"
                        ${!canDelete ? 'disabled title="Cannot delete system roles or roles with assigned users"' : ''}>
                        <svg class="w-3 h-3 mr-1 inline" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                        </svg>
                        Delete
                    </button>` : ''}
                </div>
            </div>
        </div>
    `;
}

// Render roles
function renderRoles() {
    if (!rolesContainer) return;

    // Ensure roles is an array
    if (!Array.isArray(roles)) {
        roles = [];
    }

    const searchTerm = document.getElementById('search-input')?.value.toLowerCase() || '';
    const usersFilter = document.getElementById('users-filter')?.value || '';

    let filteredRoles = roles.filter(role => {
        if (!role) return false;

        const matchesSearch = (role.name && role.name.toLowerCase().includes(searchTerm)) ||
            (role.description && role.description.toLowerCase().includes(searchTerm));

        let matchesUsers = true;
        if (usersFilter) {
            const userCount = role.users_count || 0;
            if (usersFilter === 'with-users') {
                matchesUsers = userCount > 0;
            } else if (usersFilter === 'without-users') {
                matchesUsers = userCount === 0;
            }
        }

        return matchesSearch && matchesUsers;
    });

    if (filteredRoles.length === 0) {
        rolesContainer.innerHTML = `
            <div class="col-span-full text-center py-12">
                <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z"></path>
                </svg>
                <h3 class="mt-2 text-lg font-medium text-gray-900">No roles found</h3>
                <p class="mt-1 text-sm text-gray-500">
                    ${roles.length === 0 ? 'No roles have been created yet.' : 'Try adjusting your search or filter criteria.'}
                </p>
            </div>
        `;
        return;
    }
    const editable = rolesContainer.classList.contains('editable');
    const deletable = rolesContainer.classList.contains('deletable');

    // Generate role cards
    const roleCards = filteredRoles.map(role => createRoleCard(role, editable, deletable)).filter(card => card);
    rolesContainer.innerHTML = roleCards.join('');
}

// Update statistics
function updateStatistics() {
    try {
        // Ensure roles is an array
        if (!Array.isArray(roles)) {
            roles = [];
        }

        if (!Array.isArray(permissions)) {
            permissions = [];
        }

        // Calculate total assigned users from users_count column
        const totalAssignedUsers = roles.reduce((sum, role) => {
            const userCount = Number(role.users_count) || 0;
            return sum + userCount;
        }, 0);

        const totalRolesEl = document.getElementById('total-roles');
        const activeRolesEl = document.getElementById('active-roles');
        const totalUsersEl = document.getElementById('total-assigned-users');
        const totalPermissionsEl = document.getElementById('total-permissions');

        if (totalRolesEl) totalRolesEl.textContent = roles.length;
        if (totalUsersEl) totalUsersEl.textContent = totalAssignedUsers;
        if (totalPermissionsEl) totalPermissionsEl.textContent = permissions.length;
    } catch (error) {
        console.error('Error updating statistics:', error);
    }
}

// Render permissions grid
function renderPermissionsGrid() {
    const permissionsGrid = document.getElementById('permissions-grid');
    if (!permissionsGrid) return;

    // Ensure permissions is an array
    if (!Array.isArray(permissions)) {
        permissions = [];
    }

    if (permissions.length === 0) {
        permissionsGrid.innerHTML = '<p class="text-gray-500 text-sm">No permissions available</p>';
        return;
    }

    // Group permissions by category
    const permissionsByCategory = permissions.reduce((acc, permission) => {
        if (!permission) return acc;

        const category = permission.category || 'Other';
        if (!acc[category]) {
            acc[category] = [];
        }
        acc[category].push(permission);
        return acc;
    }, {});

    let html = '';
    Object.keys(permissionsByCategory).forEach(category => {
        const categoryPermissions = permissionsByCategory[category];
        html += `
            <div class="mb-4">
                <h4 class="text-sm font-medium text-gray-700 mb-2">${category}</h4>
                ${categoryPermissions.map(permission => `
                    <label class="flex items-center mb-2 cursor-pointer">
                        <input type="checkbox"
                               class="permission-checkbox mr-2"
                               name="permissions[]"
                               value="${permission.id}">
                        <span class="text-sm text-gray-600">${formatPermission(permission.name || permission.display_name || 'Unknown')}</span>
                    </label>
                `).join('')}
            </div>
        `;
    });

    permissionsGrid.innerHTML = html;
}

// Modal functions
function openModal() {
    if (roleModal) {
        roleModal.classList.add('active');
        document.body.style.overflow = 'hidden';
    }
}

function closeModal() {
    if (roleModal) {
        roleModal.classList.remove('active');
        document.body.style.overflow = 'auto';
        resetForm();
    }
}

function resetForm() {
    if (roleForm) {
        roleForm.reset();
    }
    editingRoleId = null;
    if (modalTitle) modalTitle.textContent = 'Add New Role';
    if (submitBtn) submitBtn.textContent = 'Create Role';

    // Uncheck all permission checkboxes
    document.querySelectorAll('.permission-checkbox').forEach(checkbox => {
        checkbox.checked = false;
    });
}

// Role CRUD operations
function addRole() {
    resetForm();
    openModal();
}

function editRole(roleId) {
    // Ensure roles is an array
    if (!Array.isArray(roles)) {
        roles = [];
    }

    const role = roles.find(r => r && r.id === roleId);
    if (!role) return;

    editingRoleId = roleId;
    if (modalTitle) modalTitle.textContent = 'Edit Role';
    if (submitBtn) submitBtn.textContent = 'Update Role';

    // Populate form fields
    const nameField = document.getElementById('role-name');

    if (nameField) nameField.value = role.name || '';

    // Check appropriate permissions
    const rolePermissionIds = (role.permissions || []).map(rp => rp.id);
    document.querySelectorAll('.permission-checkbox').forEach(checkbox => {
        checkbox.checked = rolePermissionIds.includes(parseInt(checkbox.value));
    });

    openModal();
}

function deleteRole(roleId) {
    // Ensure roles is an array
    if (!Array.isArray(roles)) {
        roles = [];
    }

    const role = roles.find(r => r && r.id === roleId);
    if (!role) return;

    const canDelete = canDeleteRole(role);
    if (!canDelete) {
        showNotification('error', 'Cannot delete system roles or roles with assigned users');
        return;
    }

    if (confirm(`Are you sure you want to delete the role "${role.name}"? This action cannot be undone.`)) {
        performRoleDelete(roleId, role);
    }
}

async function performRoleDelete(roleId, role) {
    try {
        showLoading();
        await deleteRoleAPI(roleId);

        // Remove from local array
        roles = roles.filter(r => r && r.id !== roleId);

        showNotification('success', `Role "${role.name}" has been deleted successfully`);
        renderRoles();
        updateStatistics();
    } catch (error) {
        showNotification('error', `Failed to delete role: ${error.message}`);
    } finally {
        hideLoading();
    }
}

// Form submission
async function handleFormSubmission(event) {
    event.preventDefault();

    const formData = new FormData(roleForm);
    const selectedPermissions = Array.from(document.querySelectorAll('.permission-checkbox:checked'))
        .map(checkbox => parseInt(checkbox.value));

    const roleData = {
        name: formData.get('name')?.trim() || '',
        permissions: selectedPermissions
    };

    // Validation
    if (!roleData.name) {
        showNotification('error', 'Role name is required');
        return;
    }

    if (selectedPermissions.length === 0) {
        showNotification('error', 'At least one permission must be selected');
        return;
    }

    // Check for duplicate name (except when editing)
    const existingRole = roles.find(r =>
        r && r.name &&
        r.name.toLowerCase() === roleData.name.toLowerCase() &&
        r.id !== editingRoleId
    );

    if (existingRole) {
        showNotification('error', 'A role with this name already exists');
        return;
    }

    try {
        showLoading();

        if (editingRoleId) {
            // Update existing role
            const updatedRole = await updateRole(editingRoleId, roleData);
            const roleIndex = roles.findIndex(r => r && r.id === editingRoleId);
            if (roleIndex !== -1) {
                roles[roleIndex] = updatedRole;
            }
            showNotification('success', `Role "${roleData.name}" has been updated successfully`);
        } else {
            // Create new role
            const newRole = await createRole(roleData);
            if (newRole) {
                roles.push(newRole);
            }
            showNotification('success', `Role "${roleData.name}" has been created successfully`);
        }

        closeModal();
        renderRoles();
        updateStatistics();
    } catch (error) {
        showNotification('error', `Failed to ${editingRoleId ? 'update' : 'create'} role: ${error.message}`);
    } finally {
        hideLoading();
    }
}

// Initialize application
async function initializeApp() {
    try {
        showLoading();

        // Load data from API - only roles and permissions
        await Promise.all([
            fetchPermissions(),
            fetchRoles()
        ]);

        // Render initial UI
        renderPermissionsGrid();
        renderRoles();
        updateStatistics();

    } catch (error) {
        console.error('Initialization error:', error);
        showNotification('error', 'Failed to initialize application: ' + error.message);

        // Ensure arrays are initialized even on error
        if (!Array.isArray(roles)) roles = [];
        if (!Array.isArray(permissions)) permissions = [];

        // Render UI with empty data
        renderPermissionsGrid();
        renderRoles();
        updateStatistics();
    } finally {
        hideLoading();
    }
}

// Event listeners
document.addEventListener('DOMContentLoaded', function() {
    // Initialize the application
    initializeApp();

    // Button event listeners
    const addRoleBtn = document.getElementById('add-role-btn');
    const closeModalBtn = document.getElementById('close-modal');
    const cancelBtn = document.getElementById('cancel-btn');

    if (addRoleBtn) addRoleBtn.addEventListener('click', addRole);
    if (closeModalBtn) closeModalBtn.addEventListener('click', closeModal);
    if (cancelBtn) cancelBtn.addEventListener('click', closeModal);

    // Form submission
    if (roleForm) {
        roleForm.addEventListener('submit', handleFormSubmission);
    }

    // Search and filter functionality
    const searchInput = document.getElementById('search-input');
    const usersFilter = document.getElementById('users-filter');

    if (searchInput) searchInput.addEventListener('input', renderRoles);
    if (usersFilter) usersFilter.addEventListener('change', renderRoles);

    // Modal close on overlay click
    if (roleModal) {
        roleModal.addEventListener('click', function(event) {
            if (event.target === roleModal) {
                closeModal();
            }
        });
    }

    // Close modal on Escape key
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape' && roleModal?.classList.contains('active')) {
            closeModal();
        }
    });
});

// Global functions for onclick handlers
window.editRole = editRole;
window.deleteRole = deleteRole;

// Utility function to format permission names
function formatPermission(name) {
    if (!name) return 'Unknown Permission';

    return name
        .split('.')                       // split "users.edit"
        .map(word => word.charAt(0).toUpperCase() + word.slice(1)) // capitalize each part
        .join(' ');                       // join with spaces
}
