const tbody = document.getElementById('sheet-body');
const loadingIndicator = document.getElementById('loading-indicator');
const successNotification = document.getElementById('success-notification');
const errorNotification = document.getElementById('error-notification');
const dataModal = document.getElementById('data-modal');
const dataForm = document.getElementById('data-form');
let editingDataId = null;

let rowCount = 0;
let allData = []; // Store all data
let currentTab = ''; // Track current active tab
let users = []; // Store unique users

// Column resizing variables
let isResizing = false;
let currentColumn = null;
let startX = 0;
let startWidth = 0;

// Status options configuration
const statuses = [
    { text: "Service Done", color: "bg-green-500 text-white" },
    { text: "Denied", color: "bg-red-500 text-white" },
    { text: "Call Back", color: "bg-cyan-400 text-white" },
    { text: "Pitched", color: "bg-blue-600 text-white" },
    { text: "Conversation", color: "bg-yellow-400 text-black" },
    { text: "Voicemail Left", color: "bg-green-600 text-white" },
    { text: "Customer Services", color: "bg-pink-400 text-white" },
    { text: "Hung Up", color: "bg-orange-600 text-white" },
    { text: "Not Available", color: "bg-gray-400 text-white" },
    { text: "Voicemail", color: "bg-orange-500 text-white" },
    { text: "Not In Service", color: "bg-red-700 text-white" }
];

const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');

// Add additional CSS for column resizing and tabs
function addResizeCSS() {
    const additionalCSS = `
        /* Enhanced resizable columns */
        th, td {
            position: relative;
            overflow: visible;
        }

        /* Column resize handle */
        .resize-handle {
            position: absolute;
            top: 0;
            right: -2px;
            width: 4px;
            height: 100%;
            background: transparent;
            cursor: col-resize;
            user-select: none;
            z-index: 10;
            transition: background-color 0.2s;
        }

        .resize-handle:hover {
            background: #3b82f6;
        }

        .resize-handle.resizing {
            background: #1d4ed8;
        }

        /* Prevent text selection during resize */
        .no-select {
            user-select: none;
        }

        /* Enhanced status dropdown z-index */
        .status-options {
            z-index: 9999 !important;
            position: absolute;
            top: 100%;
            left: 0;
            right: 0;
            background: white;
            border: 1px solid #e5e7eb;
            border-radius: 6px;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
            max-height: 300px;
            overflow-y: auto;
            display: none;
            min-width: 150px;
        }

        /* Tab Styles */
        .tab-container {
            border-bottom: 1px solid #e5e7eb;
            background-color: #f9fafb;
        }

        .tab-nav {
            display: flex;
            overflow-x: auto;
            scrollbar-width: none;
            -ms-overflow-style: none;
        }

        .tab-nav::-webkit-scrollbar {
            display: none;
        }

        .tab-item {
            padding: 12px 20px;
            border-bottom: 3px solid transparent;
            cursor: pointer;
            white-space: nowrap;
            font-medium;
            color: #6b7280;
            transition: all 0.3s ease;
            background: none;
            border-top: none;
            border-left: none;
            border-right: none;
            outline: none;
        }

        .tab-item:hover {
            color: #374151;
            background-color: #f3f4f6;
        }

        .tab-item.active {
            color: #3b82f6;
            border-bottom-color: #3b82f6;
            background-color: white;
        }

        .loading-tabs {
            display: flex;
            gap: 1rem;
            padding: 1rem;
        }

        .loading-tab {
            height: 20px;
            width: 80px;
            background: linear-gradient(90deg, #f0f0f0 25%, #e0e0e0 50%, #f0f0f0 75%);
            background-size: 200% 100%;
            animation: loading 1.5s infinite;
            border-radius: 4px;
        }

        @keyframes loading {
            0% { background-position: 200% 0; }
            100% { background-position: -200% 0; }
        }
    `;

    const styleSheet = document.createElement('style');
    styleSheet.textContent = additionalCSS;
    document.head.appendChild(styleSheet);
}

// Initialize tabs functionality
function initUserTabs(data) {
    const tabNav = document.getElementById('tab-nav');
    const loadingTabs = document.getElementById('loading-tabs');

    // Hide loading tabs
    if (loadingTabs) {
        loadingTabs.style.display = 'none';
    }

    // Extract unique users
    users = [...new Set(data.map(item => item.user?.name || 'Unknown'))].filter(Boolean);

    // Only show tabs if there are multiple users
    if (users.length > 1) {
        // Clear existing tabs
        tabNav.innerHTML = '';

        // Add "All Users" tab
        const allTab = document.createElement('button');
        allTab.className = 'tab-item active';
        allTab.textContent = `All Users (${data.length})`;
        allTab.dataset.tab = 'all';
        allTab.addEventListener('click', () => switchTab('all'));
        tabNav.appendChild(allTab);

        // Add individual user tabs
        users.forEach(userName => {
            const userCount = data.filter(item => item.user?.name === userName).length;
            const tab = document.createElement('button');
            tab.className = 'tab-item';
            tab.textContent = `${userName} (${userCount})`;
            tab.dataset.tab = userName;
            tab.addEventListener('click', () => switchTab(userName));
            tabNav.appendChild(tab);
        });

        // Show tab container
        document.querySelector('.tab-container').style.display = 'block';
    } else {
        // Hide tab container if only one user
        document.querySelector('.tab-container').style.display = 'none';
    }
}

// Switch tab functionality
function switchTab(tabName) {
    // Update active tab
    document.querySelectorAll('.tab-item').forEach(tab => {
        tab.classList.remove('active');
        if (tab.dataset.tab === tabName) {
            tab.classList.add('active');
        }
    });

    currentTab = tabName;

    // Show/hide user column based on tab
    const userColumns = document.querySelectorAll('.user-column');
    if (tabName === 'all') {
        userColumns.forEach(col => col.style.display = '');
    } else {
        userColumns.forEach(col => col.style.display = 'none');
    }

    // Filter and display data
    const filteredData = tabName === 'all' ? allData : allData.filter(item => item.user?.name === tabName);
    displayData(filteredData);

    // Update statistics for current tab
    /*updateStatisticsForTab(filteredData);*/
}

// Display data in table
function displayData(data) {
    // Clear existing rows
    tbody.innerHTML = '';
    rowCount = 0;

    if (data.length > 0) {
        data.forEach(item => {
            createRow(item, false);
        });
    } else {
        const emptyRow = document.createElement('tr');
        const colspan = currentTab === 'all' ? '18' : '17'; // Include user column when showing all
        emptyRow.innerHTML = `
            <td colspan="${colspan}" class="p-8 text-center text-gray-500">
                <div class="flex flex-col items-center">
                    <svg class="w-12 h-12 mb-4 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                    </svg>
                    <p class="text-lg font-medium">No records found</p>
                </div>
            </td>
        `;
        tbody.appendChild(emptyRow);
    }
}

// Initialize column resizing functionality
function initColumnResize() {
    addResizeCSS();

    const table = document.getElementById('sheet');
    table.style.tableLayout = 'fixed';

    if(table.classList.contains('all') && currentTab === '') {
        switchTab('all');
    }

    addResizeHandlesToHeaders();
    setupResizeEventListeners();
    setInitialColumnWidths();
}

function addResizeHandlesToHeaders() {
    const headerCells = document.querySelectorAll('#sheet thead th');

    headerCells.forEach((th, index) => {
        if (index < headerCells.length - 1) {
            const resizeHandle = document.createElement('div');
            resizeHandle.className = 'resize-handle';
            th.appendChild(resizeHandle);
        }
    });
}

function setInitialColumnWidths() {
    const headerCells = document.querySelectorAll('#sheet thead th');
    const columnWidths = [
        '60px',   // S.No
        '120px',  // User
        '100px',  // Date
        '100px',  // Country
        '150px',  // Customer Name
        '140px',  // Dialing Status
        '200px',  // Comments
        '150px',  // Time & Date
        '200px',  // Email
        '120px',  // Number
        '100px',  // Call Type
        '120px',  // Company
        '120px',  // Last Amount
        '80px',   // BTTR
        '130px',  // Payment Date
        '140px',  // Final Status
        '120px',  // Agent Name
        '100px'   // Action
    ];

    headerCells.forEach((th, index) => {
        if (columnWidths[index]) {
            th.style.width = columnWidths[index];
            th.style.minWidth = columnWidths[index];
        }
    });

    updateBodyCellWidths();
}

function updateBodyCellWidths() {
    const headerCells = document.querySelectorAll('#sheet thead th');
    const bodyRows = document.querySelectorAll('#sheet tbody tr');

    bodyRows.forEach(row => {
        const cells = row.querySelectorAll('td');
        cells.forEach((cell, index) => {
            if (headerCells[index]) {
                const width = headerCells[index].style.width;
                if (width) {
                    cell.style.width = width;
                    cell.style.minWidth = width;
                }
            }
        });
    });
}

function setupResizeEventListeners() {
    document.addEventListener('mousedown', handleMouseDown);
    document.addEventListener('mousemove', handleMouseMove);
    document.addEventListener('mouseup', handleMouseUp);

    document.addEventListener('contextmenu', (e) => {
        if (e.target.classList.contains('resize-handle')) {
            e.preventDefault();
        }
    });
}

function handleMouseDown(e) {
    const resizeHandle = e.target.closest('.resize-handle');
    if (!resizeHandle) return;

    isResizing = true;
    currentColumn = resizeHandle.parentElement;
    startX = e.clientX;
    startWidth = parseInt(window.getComputedStyle(currentColumn).width, 10);

    resizeHandle.classList.add('resizing');
    document.body.classList.add('no-select');
    document.body.style.cursor = 'col-resize';

    e.preventDefault();
    e.stopPropagation();
}

function handleMouseMove(e) {
    if (!isResizing || !currentColumn) return;

    const diff = e.clientX - startX;
    const newWidth = Math.max(50, startWidth + diff);

    currentColumn.style.width = newWidth + 'px';
    currentColumn.style.minWidth = newWidth + 'px';

    updateCorrespondingBodyCells(currentColumn, newWidth);

    e.preventDefault();
}

function handleMouseUp(e) {
    if (!isResizing) return;

    isResizing = false;
    currentColumn = null;

    document.body.classList.remove('no-select');
    document.body.style.cursor = '';

    document.querySelectorAll('.resize-handle.resizing').forEach(handle => {
        handle.classList.remove('resizing');
    });

    e.preventDefault();
}

function updateCorrespondingBodyCells(headerCell, width) {
    const headerCells = Array.from(document.querySelectorAll('#sheet thead th'));
    const columnIndex = headerCells.indexOf(headerCell);

    if (columnIndex === -1) return;

    const bodyRows = document.querySelectorAll('#sheet tbody tr');
    bodyRows.forEach(row => {
        const cell = row.children[columnIndex];
        if (cell) {
            cell.style.width = width + 'px';
            cell.style.minWidth = width + 'px';
        }
    });
}

function addResizeHandlesToNewRow(row) {
    const headerCells = document.querySelectorAll('#sheet thead th');
    const cells = row.querySelectorAll('td');

    cells.forEach((cell, index) => {
        if (headerCells[index]) {
            const width = headerCells[index].style.width;
            if (width) {
                cell.style.width = width;
                cell.style.minWidth = width;
            }
        }
    });
}

// Status dropdown functionality
function initStatusDropdowns() {
    document.addEventListener('click', (e) => {
        const selectedElement = e.target.closest('.status-selected');
        if (selectedElement) {
            e.stopPropagation();
            document.querySelectorAll('.status-options').forEach(options => {
                if (options !== selectedElement.nextElementSibling) {
                    options.style.display = 'none';
                }
            });

            const optionsDiv = selectedElement.nextElementSibling;
            optionsDiv.style.display = optionsDiv.style.display === 'block' ? 'none' : 'block';
            return;
        }

        const optionElement = e.target.closest('.status-option');
        if (optionElement) {
            const selectedDiv = optionElement.closest('.status-dropdown').querySelector('.status-selected');
            const optionsDiv = optionElement.parentElement;

            selectedDiv.className = optionElement.className.replace('status-option', 'status-selected');
            selectedDiv.textContent = optionElement.textContent;

            optionsDiv.style.display = 'none';

            console.log('Status changed to:', optionElement.textContent);
            return;
        }

        document.querySelectorAll('.status-options').forEach(options => {
            options.style.display = 'none';
        });
    });
}

// Utility functions
function showLoading() {
    if (loadingIndicator) {
        loadingIndicator.style.display = 'block';
    }
}

function hideLoading() {
    if (loadingIndicator) {
        loadingIndicator.style.display = 'none';
    }
}

function showNotification(type, message) {
    if (!successNotification || !errorNotification) return;

    const notification = type === 'success' ? successNotification : errorNotification;
    const messageElement = type === 'success' ?
        document.getElementById('success-message') :
        document.getElementById('error-message');

    if (messageElement) {
        messageElement.textContent = message;
    }
    notification.style.display = 'block';

    setTimeout(() => {
        notification.style.display = 'none';
    }, 4000);
}

async function makeRequest(url, method, data = null) {
    showLoading();
    try {
        const options = {
            method: method,
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': csrfToken,
                'Accept': 'application/json'
            }
        };

        if (data) {
            options.body = JSON.stringify(data);
        }

        const response = await fetch(url, options);
        const result = await response.json();

        if (!response.ok) {
            throw new Error(result.message || 'Request failed');
        }

        return result;
    } catch (error) {
        console.error('Request error:', error);
        showNotification('error', error.message);
        throw error;
    } finally {
        hideLoading();
    }
}

// Create custom status dropdown
function createStatusDropdown(fieldType, selected = "", rowId = null, editable = false) {
    const dropdown = document.createElement('div');
    dropdown.className = 'status-dropdown';

    const selectedDiv = document.createElement('div');
    const selectedStatus = statuses.find(s => s.text === selected) || statuses[0];
    selectedDiv.className = `status-selected ${selectedStatus.color}`;
    selectedDiv.textContent = selectedStatus.text;

    if(editable) {
        const optionsDiv = document.createElement('div');
        optionsDiv.className = 'status-options';

        statuses.forEach(status => {
            const option = document.createElement('div');
            option.className = `status-option ${status.color}`;
            option.textContent = status.text;
            option.addEventListener('click', async () => {
                const oldStatus = selectedDiv.textContent;
                selectedDiv.className = `status-selected ${status.color}`;
                selectedDiv.textContent = status.text;
                optionsDiv.style.display = 'none';

                if (rowId) {
                    try {
                        await makeRequest(`/custom-data/${rowId}`, 'PUT', {
                            [fieldType]: status.text
                        });
                        showNotification('success', 'Status updated successfully');
                        /*updateStatistics();*/
                    } catch (error) {
                        const revertStatus = statuses.find(s => s.text === oldStatus) || statuses[0];
                        selectedDiv.className = `status-selected ${revertStatus.color}`;
                        selectedDiv.textContent = revertStatus.text;
                    }
                }
            });
            optionsDiv.appendChild(option);
        });

        selectedDiv.addEventListener('click', (e) => {
            e.stopPropagation();
            document.querySelectorAll('.status-options').forEach(opt => {
                if (opt !== optionsDiv) opt.style.display = 'none';
            });
            optionsDiv.style.display = optionsDiv.style.display === 'block' ? 'none' : 'block';
        });

        dropdown.appendChild(selectedDiv);
        dropdown.appendChild(optionsDiv);
    }

    dropdown.appendChild(selectedDiv);

    return dropdown;
}

// Create editable field
function createEditableField(content = "", fieldName = "", rowId = null, editable = false) {
    const div = document.createElement('div');
    div.className = `${editable ? 'editable ' : ''}min-h-[32px] px-2 py-1 focus:outline-none`;
    if(editable) {
        div.contentEditable = 'true';
    }
    div.textContent = content;

    let originalValue = content;

    div.addEventListener('focus', () => {
        originalValue = div.textContent.trim();
    });

    if (rowId) {
        div.addEventListener('focusout', async () => {
            const newValue = div.textContent.trim();
            if (newValue !== originalValue) {
                try {
                    await makeRequest(`/custom-data/${rowId}`, 'PUT', {
                        [fieldName]: newValue
                    });
                    showNotification('success', 'Field updated successfully');
                    originalValue = newValue;
                } catch (error) {
                    div.textContent = originalValue;
                }
            }
        });
    }

    return div;
}

// Create table row
function createRow(data = {}, isNew = true) {
    rowCount++;
    const tr = document.createElement('tr');
    tr.className = 'hover:bg-gray-50 transition-colors';
    tr.dataset.id = data.id || '';

    const fields = [
        { name: 'S.No', key: 'id', editable: false },
        { name: 'user', key: 'user', editable: false },
        { name: 'date', key: 'date', editable: false },
        { name: 'country', key: 'country', editable: false },
        { name: 'customer_name', key: 'customer_name', editable: false },
        { name: 'daily_status', key: 'daily_status', editable: true, type: 'status' },
        { name: 'comments', key: 'comments', editable: true },
        { name: 'time_date', key: 'time_date', editable: true },
        { name: 'email', key: 'email', editable: false },
        { name: 'number', key: 'number', editable: false },
        { name: 'call_type', key: 'call_type', editable: false },
        { name: 'company', key: 'company', editable: false },
        { name: 'last_amount', key: 'last_amount', editable: false },
        { name: 'bttr', key: 'bttr', editable: false },
        { name: 'payment_date', key: 'payment_date', editable: false },
        { name: 'final_status', key: 'final_status', editable: true, type: 'status' },
        { name: 'agent_name', key: 'agent_name', editable: false },
        { name: 'action', key: 'action', editable: false }
    ];

    const editable = tbody.classList.contains('editable');
    const editableClass = editable ? '' : 'non-editable';
    const deletable = tbody.classList.contains('deletable');

    fields.forEach((field) => {
        const td = document.createElement('td');
        td.className = 'p-3 text-sm border-r border-b';

        if (field.name === 'S.No') {
            td.classList.add('serial-col');
            td.textContent = rowCount;
            td.className += ` ${editableClass} font-medium text-gray-600`;
        } else if (field.name === 'user') {
            td.classList.add('user-column');
            td.textContent = data.user?.name || 'Unknown';
            td.className += ` ${editableClass}`;
            // Hide user column by default (will be shown when "All Users" tab is active)
            if (currentTab !== 'all') {
                td.style.display = 'none';
            }
        } else if (field.type === 'status') {
            td.appendChild(createStatusDropdown(field.key, data[field.key] || "", data.id, editable));
        } else if (field.editable) {
            td.appendChild(createEditableField(data[field.key] || '', field.key, data.id, editable));
        } else if (field.name === 'action') {
            if(editable) {
                const editBtn = document.createElement('button');
                editBtn.innerHTML = `
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z"></path>
                    </svg>
                `;
                editBtn.className = 'px-2 py-1 bg-yellow-600 text-white rounded hover:bg-yellow-700 transition-colors mr-2';
                editBtn.title = 'Edit Row';
                editBtn.addEventListener('click', async () => {
                    editData(data.id)
                });
                td.appendChild(editBtn);
            }
            if(deletable) {
                const deleteBtn = document.createElement('button');
                    deleteBtn.innerHTML = `
                    <svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16"></path>
                    </svg>
                `;
                deleteBtn.className = 'px-2 py-1 bg-red-600 text-white rounded hover:bg-red-700 transition-colors';
                deleteBtn.title = 'Delete Row';
                deleteBtn.addEventListener('click', async () => {
                    if (confirm('Are you sure you want to delete this row?')) {
                        if (data.id) {
                            try {
                                await makeRequest(`/custom-data/${data.id}`, 'DELETE');
                                showNotification('success', 'Record deleted successfully');

                                // Remove from allData array
                                allData = allData.filter(item => item.id !== data.id);

                                // Update tabs with new counts
                                initUserTabs(allData);

                                // Refresh current tab display
                                switchTab(currentTab);

                                /*updateStatistics();*/
                            } catch (error) {
                                return;
                            }
                        } else {
                            // Remove from allData array for non-persisted rows
                            allData = allData.filter(item => item.id !== data.id);
                            tr.remove();
                            updateSerialNumbers();
                        }
                    }
                });
                td.appendChild(deleteBtn);
            }
        } else {
            const value = data[field.key];
            if (field.key === 'last_amount' && value) {
                td.textContent = parseFloat(value).toFixed(2);
            } else if (field.key === 'date' || field.key === 'payment_date') {
                td.textContent = value ? value : '';
            } else if (field.key === 'time_date') {
                td.textContent = value ? new Date(value).toLocaleString() : '';
            } else {
                td.textContent = value || '';
            }
            td.className += ` ${editableClass}`;
        }

        tr.appendChild(td);
    });

    tbody.appendChild(tr);

    // Apply column widths to new row
    addResizeHandlesToNewRow(tr);

    return tr;
}

// Update serial numbers
function updateSerialNumbers() {
    let sn = 1;
    tbody.querySelectorAll('tr:not(.skeleton-loading)').forEach(row => {
        const firstCell = row.querySelector('td');
        if (firstCell && firstCell.classList.contains('serial-col')) {
            firstCell.textContent = sn++;
        }
    });
    rowCount = sn - 1;
}

// Hide skeleton loading
function hideSkeletonLoading() {
    document.querySelectorAll('.skeleton-loading').forEach(row => {
        row.style.display = 'none';
    });
}

// Update statistics
/*async function updateStatistics() {
    try {
        const response = await fetch('/custom-data/stats/dashboard');
        const result = await response.json();

        if (result.success) {
            document.getElementById('total-records').textContent = result.data.total_records;

            const salesDone = result.data.final_status_distribution.find(item => item.final_status === 'Sale Done');
            document.getElementById('sales-done').textContent = salesDone ? salesDone.count : 0;

            const pendingCalls = result.data.daily_status_distribution.find(item => item.daily_status === 'Call Back');
            document.getElementById('pending-calls').textContent = pendingCalls ? pendingCalls.count : 0;
        }
    } catch (error) {
        console.error('Error updating statistics:', error);
    }
}*/

// Update statistics for specific tab data
/*function updateStatisticsForTab(data) {
    const totalRecords = data.length;

    const salesDone = data.filter(item => item.final_status === 'Sale Done').length;
    const pendingCalls = data.filter(item => item.daily_status === 'Call Back').length;

    // Update DOM if elements exist
    const totalElement = document.getElementById('total-records');
    const salesElement = document.getElementById('sales-done');
    const pendingElement = document.getElementById('pending-calls');

    if (totalElement) totalElement.textContent = totalRecords;
    if (salesElement) salesElement.textContent = salesDone;
    if (pendingElement) pendingElement.textContent = pendingCalls;
}*/

async function loadData() {
    try {
        const response = await makeRequest('/custom-data', 'GET');
        hideSkeletonLoading();

        if (response.success && response.data && response.data.length > 0) {
            allData = response.data;

            // Initialize tabs based on data
            initUserTabs(allData);

            // Display all data initially
            displayData(allData);
        } else {
            allData = [];
            // Hide tabs if no data
            document.querySelector('.tab-container').style.display = 'none';

            const emptyRow = document.createElement('tr');
            emptyRow.innerHTML = `
                <td colspan="17" class="p-8 text-center text-gray-500">
                    <div class="flex flex-col items-center">
                        <svg class="w-12 h-12 mb-4 text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z"></path>
                        </svg>
                        <p class="text-lg font-medium">No records found</p>
                    </div>
                </td>
            `;
            tbody.appendChild(emptyRow);
        }

        /*updateStatistics();*/
    } catch (error) {
        hideSkeletonLoading();
        showNotification('error', 'Failed to load data');
    }
}


// Modal functions
function openModal() {
    if (dataModal) {
        dataModal.classList.add('active');
        document.body.style.overflow = 'hidden';
    }
}

function closeModal() {
    if (dataModal) {
        dataModal.classList.remove('active');
        document.body.style.overflow = 'auto';
        resetForm();
    }
}

function resetForm() {
    if (dataForm) {
        dataForm.reset();
    }

}

// User CRUD operations
function addUser() {
    resetForm();
    openModal();
}

// Form submission
async function handleFormSubmission(event) {
    event.preventDefault();

    // Use the correct form ID from the HTML
    const form = document.getElementById('user-form');
    const formData = new FormData(form);

    // Map form fields to match the HTML form structure
    const customData = {
        date: formData.get('date')?.trim() || '',
        country: formData.get('country')?.trim() || '',
        customer_name: formData.get('customer_name')?.trim() || '',
        daily_status: formData.get('daily_status')?.trim() || '',
        comments: formData.get('comments')?.trim() || '',
        time_date: formData.get('time_date')?.trim() || '',
        email: formData.get('email')?.trim() || '',
        number: formData.get('number')?.trim() || '',
        call_type: formData.get('call_type')?.trim() || '',
        company: formData.get('company')?.trim() || '',
        last_amount: formData.get('last_amount')?.trim() || '',
        bttr: formData.get('bttr')?.trim() || '',
        payment_date: formData.get('payment_date')?.trim() || '',
        final_status: formData.get('final_status')?.trim() || '',
        agent_name: formData.get('agent_name')?.trim() || ''
    };

    // Basic validation for required fields
    if (!customData.date) {
        showNotification('error', 'Date is required');
        return;
    }

    if (!customData.country) {
        showNotification('error', 'Country is required');
        return;
    }

    if (!customData.customer_name) {
        showNotification('error', 'Customer Name is required');
        return;
    }

    if (!customData.daily_status) {
        showNotification('error', 'Dialing Status is required');
        return;
    }

    if (!customData.email) {
        showNotification('error', 'Email is required');
        return;
    }

    if (!customData.number) {
        showNotification('error', 'Number is required');
        return;
    }

    if (!customData.company) {
        showNotification('error', 'Company is required');
        return;
    }
    if (typeof Number(customData.last_amount) !== 'number') {
        showNotification('error', 'Last Amount should be a number');
        return;
    }

    try {
        showLoading();

        let result;
        if (editingDataId) {
            // Update existing record
            result = await makeRequest(`/custom-data/${editingDataId}`, 'PUT', customData);

            if (result.success) {
                showNotification('success', 'Data record updated successfully');

                // Update the record in allData array
                const index = allData.findIndex(item => item.id === editingDataId);
                if (index !== -1) {
                    allData[index] = { ...allData[index], ...result.data };
                }
            }
        } else {
            // Create new record
            result = await makeRequest('/custom-data', 'POST', customData);

            if (result.success) {
                showNotification('success', 'Data record created successfully');

                // Add new record to allData array
                allData.unshift(result.data);
            }
        }

        if (result.success) {
            // Update tabs with new/updated data
            initUserTabs(allData);

            // Refresh current tab display
            switchTab(currentTab || 'all');

            closeModal();
        } else {
            throw new Error(result.message || `Failed to ${editingDataId ? 'update' : 'create'} record`);
        }
    } catch (error) {
        showNotification('error', `Failed to ${editingDataId ? 'update' : 'create'} record: ` + error.message);
    } finally {
        hideLoading();
    }
}

function resetForm() {
    const form = document.getElementById('user-form');
    if (form) {
        form.reset();
    }
    editingDataId = null;
}

// Function to populate form with existing data for editing
function populateFormForEdit(data) {
    const form = document.getElementById('user-form');
    if (!form || !data) return;

    // Populate all form fields
    const dateField = form.querySelector('#date');
    const countryField = form.querySelector('#country');
    const customerNameField = form.querySelector('#customer_name');
    const dailyStatusField = form.querySelector('#daily_status');
    const commentsField = form.querySelector('#comments');
    const timeDateField = form.querySelector('#time_date');
    const emailField = form.querySelector('#email');
    const numberField = form.querySelector('#number');
    const callTypeField = form.querySelector('#call_type');
    const companyField = form.querySelector('#company');
    const lastAmountField = form.querySelector('#last_amount');
    const bttrField = form.querySelector('#bttr');
    const paymentDateField = form.querySelector('#payment_date');
    const finalStatusField = form.querySelector('#final_status');
    const agentNameField = form.querySelector('#agent_name');

    if (dateField) dateField.value = data.date || '';
    if (countryField) countryField.value = data.country || '';
    if (customerNameField) customerNameField.value = data.customer_name || '';
    if (dailyStatusField) dailyStatusField.value = data.daily_status || '';
    if (commentsField) commentsField.value = data.comments || '';
    if (timeDateField) timeDateField.value = data.time_date || '';
    if (emailField) emailField.value = data.email || '';
    if (numberField) numberField.value = data.number || '';
    if (callTypeField) callTypeField.value = data.call_type || '';
    if (companyField) companyField.value = data.company || '';
    if (lastAmountField) lastAmountField.value = data.last_amount || '';
    if (bttrField) bttrField.value = data.bttr || '';
    if (paymentDateField) paymentDateField.value = data.payment_date || '';
    if (finalStatusField) finalStatusField.value = data.final_status || '';
    if (agentNameField) agentNameField.value = data.agent_name || '';
}
function editData(dataId) {
    const data = allData.find(item => item.id === dataId);
    if (!data) {
        showNotification('error', 'Record not found');
        return;
    }

    editingDataId = dataId;

    // Update modal title
    const modalTitle = document.getElementById('modal-title');
    const submitBtn = document.getElementById('submit-btn');

    if (modalTitle) modalTitle.textContent = 'Edit Data Row';
    if (submitBtn) submitBtn.textContent = 'Update Record';

    // Populate form with existing data
    populateFormForEdit(data);

    // Open modal
    openModal();
}

// Function to add new data (update existing addUser function)
function addData() {
    editingDataId = null;

    // Update modal title
    const modalTitle = document.getElementById('modal-title');
    const submitBtn = document.getElementById('submit-btn');

    if (modalTitle) modalTitle.textContent = 'Add New Data Row';
    if (submitBtn) submitBtn.textContent = 'Create Record';

    resetForm();
    openModal();
}

// Event listeners
document.addEventListener('DOMContentLoaded', function () {
    setTimeout(() => {
        initColumnResize();
        initStatusDropdowns();
        loadData();

        // Button event listeners
        const addUserBtn = document.getElementById('add-row-btn');
        const closeModalBtn = document.getElementById('close-modal');
        const cancelBtn = document.getElementById('cancel-btn');

        if (addUserBtn) addUserBtn.addEventListener('click', addData); // Changed from addUser to addData
        if (closeModalBtn) closeModalBtn.addEventListener('click', closeModal);
        if (cancelBtn) cancelBtn.addEventListener('click', closeModal);

        // Form submission - use the correct form ID
        const form = document.getElementById('user-form');
        if (form) {
            form.addEventListener('submit', handleFormSubmission);
        }

        // Modal close on overlay click
        if (dataModal) {
            dataModal.addEventListener('click', function(event) {
                if (event.target === dataModal) {
                    closeModal();
                }
            });
        }

        // Close modal on Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape' && dataModal?.classList.contains('active')) {
                closeModal();
            }
        });
    }, 100);
});

// Button event listeners
if (document.getElementById('import-data')) {
    document.getElementById('import-data').addEventListener('click', () => {
        document.getElementById('file-input').click();
    });
}

if (document.getElementById('export-data')) {
    document.getElementById('export-data').addEventListener('click', async () => {
        try {
            const response = await makeRequest('/custom-data/export/csv', 'GET');
            console.log(response);
            if (response.success) {
                const blob = new Blob([response.data], { type: 'text/csv' });
                const url = window.URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `custom_data_export_${new Date().toISOString().split('T')[0]}.csv`;
                a.click();
                window.URL.revokeObjectURL(url);
                showNotification('success', 'Data exported successfully');
            }
        } catch (error) {
            showNotification('error', 'Failed to export data');
        }
    });
}

if (document.getElementById('file-input')) {
    document.getElementById('file-input').addEventListener('change', async (e) => {
        const file = e.target.files[0];
        if (file) {
            const formData = new FormData();
            formData.append('file', file);

            try {
                showLoading();
                const response = await fetch('/custom-data/import', {
                    method: 'POST',
                    body: formData,
                    headers: { 'X-CSRF-TOKEN': csrfToken }
                });

                const result = await response.json();

                if (response.ok && result.success) {
                    showNotification('success', result.message);
                    setTimeout(() => { location.reload(); }, 1500);
                } else {
                    throw new Error(result.message || 'Import failed');
                }
            } catch (error) {
                showNotification('error', 'Import failed: ' + error.message);
            } finally {
                hideLoading();
                e.target.value = '';
            }
        }
    });
}

// Close dropdowns when clicking outside
document.addEventListener('click', (e) => {
    if (!e.target.closest('.status-dropdown')) {
        document.querySelectorAll('.status-options').forEach(options => {
            options.style.display = 'none';
        });
    }
});

// Export utility functions for external use
window.DashboardUtils = {
    createRow,
    showNotification,
    makeRequest,
    switchTab,
    displayData,
    editData,  // Added editData function
    addData,   // Added addData function
    users,
    allData
};
