<?php

namespace App\Http\Controllers;

use App\Models\CustomData;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Spatie\Permission\Models\Role;

class CustomDataController extends Controller
{
    public function index()
    {
        if (Gate::allows('viewAll', CustomData::class)) {
            Log::info('User can view all records');
            $query = CustomData::query();
        } elseif (auth()->user()->can('data.view')) {
            $query = CustomData::where('user_id', auth()->id());
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        try {
            $data = $query->with(['user'])->orderBy('created_at', 'desc')->get();

            return response()->json([
                'success' => true,
                'data' => $data,
                'message' => 'Data retrieved successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving data: ' . $e->getMessage()
            ], 500);
        }
    }


    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        Gate::authorize('create', CustomData::class);
        $validator = Validator::make($request->all(), [
            'date' => 'nullable|date',
            'country' => 'nullable|string|max:100',
            'customer_name' => 'nullable|string|max:150',
            'daily_status' => 'nullable|string|max:255',
            'comments' => 'nullable|string',
            'time_date' => 'nullable|date',
            'email' => 'nullable|email|max:150',
            'number' => 'nullable|string|max:50',
            'call_type' => 'nullable|string|max:100',
            'company' => 'nullable|string|max:150',
            'last_amount' => 'nullable|numeric|min:0',
            'bttr' => 'nullable|string|max:100',
            'payment_date' => 'nullable|date',
            'final_status' => 'nullable|string|max:255',
            'status' => 'nullable|string|max:100',
            'agent_name' => 'nullable|string|max:150',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
                'message' => 'Validation failed'
            ], 422);
        }

        try {

            $data = CustomData::create(
                array_merge(
                    ['user_id' => Auth::id()],
                    $request->all()
                )
            );
            $data->load('user');

            return response()->json([
                'success' => true,
                'data' => $data,
                'message' => 'Record created successfully'
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating record: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        Gate::authorize('view', CustomData::class);
        try {
            $data = CustomData::findOrFail($id);

            return response()->json([
                'success' => true,
                'data' => $data,
                'message' => 'Record retrieved successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Record not found'
            ], 404);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        Gate::authorize('edit', CustomData::class);
        $validator = Validator::make($request->all(), [
            'date' => 'nullable|date',
            'country' => 'nullable|string|max:100',
            'customer_name' => 'nullable|string|max:150',
            'daily_status' => 'nullable|string|max:255',
            'comments' => 'nullable|string',
            'time_date' => 'nullable|date',
            'email' => 'nullable|email|max:150',
            'number' => 'nullable|string|max:50',
            'call_type' => 'nullable|string|max:100',
            'company' => 'nullable|string|max:150',
            'last_amount' => 'nullable|numeric|min:0',
            'bttr' => 'nullable|string|max:100',
            'payment_date' => 'nullable|date',
            'final_status' => 'nullable|string|max:255',
            'status' => 'nullable|string|max:100',
            'agent_name' => 'nullable|string|max:150',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
                'message' => 'Validation failed'
            ], 422);
        }

        try {
            $data = CustomData::findOrFail($id);
            $data->update($request->all());

            return response()->json([
                'success' => true,
                'data' => $data,
                'message' => 'Record updated successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating record: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        Gate::authorize('delete', CustomData::class);
        try {
            $data = CustomData::findOrFail($id);
            $data->delete();

            return response()->json([
                'success' => true,
                'message' => 'Record deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting record: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Export data to CSV
     */
    // public function export()
    // {
    //     try {
    //         $data = CustomData::all();
    //         $csvData = "ID,DATE,COUNTRY,CUSTOMERS NAMES,DIALING STATUS,COMMENTS,TIME & DATE,EMAIL,NUMBER,CALL TYPE,COMPANY,LAST AMOUNT,BTTR,PAYMENT DATES,FINAL STATUS,AGENT NAME";

    //         foreach ($data as $row) {
    //             $csvData .= sprintf(
    //                 "%d,%s,%s,%s,%s,\"%s\",%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s,%s\n",
    //                 $row->id,
    //                 $row->date ?? '',
    //                 $row->country ?? '',
    //                 $row->customer_name ?? '',
    //                 $row->daily_status ?? '',
    //                 str_replace(['"', "\n", "\r"], ['""', ' ', ' '], $row->comments ?? ''),
    //                 $row->time_date ?? '',
    //                 $row->email ?? '',
    //                 $row->number ?? '',
    //                 $row->call_type ?? '',
    //                 $row->company ?? '',
    //                 $row->last_amount ?? '',
    //                 $row->bttr ?? '',
    //                 $row->payment_date ?? '',
    //                 $row->final_status ?? '',
    //                 $row->status ?? '',
    //                 $row->agent_name ?? ''
    //             );
    //         }

    //         return response()->json([
    //             'success' => true,
    //             'data' => $csvData,
    //             'message' => 'Data exported successfully'
    //         ]);
    //     } catch (\Exception $e) {
    //         return response()->json([
    //             'success' => false,
    //             'message' => 'Error exporting data: ' . $e->getMessage()
    //         ], 500);
    //     }
    // }


    public function export(Request $request)
    {
        if (Gate::allows('exportAll', CustomData::class)) {
            Log::info('User can export all records');
            if($request->user_id) {
                $query = CustomData::where('user_id', $request->user_id);
            } else {
                $query = CustomData::query();
            }
        } elseif (auth()->user()->can('data.export')) {
            $query = CustomData::where('user_id', $request->user_id ? $request->user_id : auth()->id());
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized'
            ], 403);
        }

        try {
            $data = $query->orderBy('created_at', 'desc')->get();

            // CSV header matching your exact requirements
            $csvData = "ID,DATE,COUNTRY,CUSTOMERS NAMES,DIALING STATUS,COMMENTS,TIME & DATE,EMAIL,NUMBER,CALL TYPE,COMPANY,LAST AMOUNT,BTTR,PAYMENT DATE,FINAL STATUS,AGENT NAME\n";

            foreach ($data as $row) {
                $csvData .= sprintf(
                    "%d,%s,%s,%s,%s,\"%s\",%s,%s,%s,%s,%s,%s,%s,%s,%s,%s\n",
                    $row->id,
                    $row->date ?? '',
                    $row->country ?? '',
                    $row->customer_name ?? '',
                    $row->daily_status ?? '',
                    str_replace(['"', "\n", "\r"], ['""', ' ', ' '], $row->comments ?? ''),
                    $row->time_date ?? '',
                    $row->email ?? '',
                    $row->number ?? '',
                    $row->call_type ?? '',
                    $row->company ?? '',
                    $row->last_amount ?? '',
                    $row->bttr ?? '',
                    $row->payment_date ?? '',
                    $row->final_status ?? '',
                    $row->agent_name ?? ''
                );
            }

            return response()->json([
                'success' => true,
                'data' => $csvData,
                'message' => 'Data exported successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error exporting data: ' . $e->getMessage()
            ], 500);
        }
    }

    public function import(Request $request)
    {
        Gate::authorize('haveAnyImport', CustomData::class);
        $validator = Validator::make($request->all(), [
            'file' => 'required|file|mimes:csv,txt,xlsx,xls|max:10240',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
                'message' => 'Invalid file format'
            ], 422);
        }

        try {
            $file = $request->file('file');
            $extension = $file->getClientOriginalExtension();

            if(Gate::allows('importAll', CustomData::class)) {
                $user = $request->user_id ? User::find($request->user_id) : Auth::user();
            } elseif (auth()->user()->can('data.import')) {
                $user = User::find(auth()->id());
            }

            if ($extension === 'csv') {
                return $this->importCsv($file, $user);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'Excel import requires Laravel Excel package installation'
                ], 501);
            }
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error importing file: ' . $e->getMessage()
            ], 500);
        }
    }


    private function importCsv($file, $user)
    {
        $csvData = file_get_contents($file->getPathname());
        $lines   = explode("\n", $csvData);
        $header  = str_getcsv(array_shift($lines));

        $importedCount = 0;
        $errors = [];

        DB::beginTransaction();

        try {
            foreach ($lines as $lineNumber => $line) {
                if (empty(trim($line))) continue;

                $row = str_getcsv($line);

                if (count($row) !== count($header)) {
                    $errors[] = sprintf(
                        "Line %d: Column count mismatch. Expected %d, got %d. Row: %s",
                        $lineNumber + 2,
                        count($header),
                        count($row),
                        json_encode($row)
                    );
                    continue;
                }

                $data = array_combine($header, $row);

                CustomData::create([
                    'user_id'      => $user->id,
                    'date'         => !empty($data['DATE']) ? $data['DATE'] : null,
                    'country'      => $data['COUNTRY'] ?? null,
                    'customer_name' => $data['CUSTOMERS NAMES'] ?? null,
                    'daily_status' => $data['DIALING STATUS'] ?? null,
                    'comments'     => $data['COMMENTS'] ?? null,
                    'time_date'    => !empty($data['TIME & DATE']) ? $data['TIME & DATE'] : null,
                    'email'        => $data['EMAIL'] ?? null,
                    'number'       => $data['NUMBER'] ?? null,
                    'call_type'    => $data['CALL TYPE'] ?? null,
                    'company'      => $data['COMPANY'] ?? null,
                    'last_amount'  => !empty($data['LAST AMOUNT']) ? floatval(str_replace(['$', ','], '', $data['LAST AMOUNT'])) : null,
                    'bttr'         => $data['BTTR'] ?? null,
                    'payment_date' => !empty($data['PAYMENT DATES']) ? $data['PAYMENT DATES'] : null,
                    'final_status' => $data['FINAL STATUS'] ?? null,
                    'agent_name'   => $data['AGENT NAME'] ?? null,
                ]);

                $importedCount++;
            }
            if (!empty($errors)) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => "Import failed. Nothing was saved.",
                    'imported_count' => $importedCount,
                    'errors' => $errors
                ], 500);
            }
            DB::commit();
            return response()->json([
                'success' => true,
                'message' => "Successfully imported {$importedCount} records",
                'imported_count' => $importedCount,
                'errors' => []
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => "Import failed: " . $e->getMessage(),
                'imported_count' => $importedCount,
                'errors' => $errors
            ], 400);
        }
    }

    public function checkPermissions()
    {
        return response()->json([
            'success' => true,
            'data' => [
                'canViewAll' => Gate::allows('viewAll', CustomData::class),
                'canView'    => auth()->user()->can('data.view'),
            ]
        ]);
    }
}
